#include "machtalk_link.h"

#ifdef ESP8266
#define _tag __attribute__((section(".irom0.text")))
#else
#define _tag  
#endif

#ifdef __WIN__
#define _MTPACKED 
#else
#define _MTPACKED __attribute__((packed))
#endif

#define BSSID_RECODR_NUM 5

typedef struct FrameCtrl80211_st
{
    unsigned char Protocol : 2;
    unsigned char type : 2;
    unsigned char subtype : 4;
    unsigned char toDS : 1;
    unsigned char FromDS : 1;
    unsigned char MoreFlag : 1;
    unsigned char Retry : 1;
    unsigned char PwrMgmt : 1;
    unsigned char MoreData : 1;
    unsigned char Protectedframe : 1;
    unsigned char Order : 1;
}_MTPACKED FrameCtrl80211_t; 

typedef struct IEEE80211DataFrame_st
{
    FrameCtrl80211_t framectrl;
    unsigned short duration;
    unsigned char raddr[6];
    unsigned char taddr[6];
    unsigned char saddr[6];
    unsigned short seq;
    unsigned char data[];
}_MTPACKED IEEE80211DataFrame_t;

typedef struct
{
    unsigned char bssid[6];
    unsigned char sn;
    unsigned short len_offset[2];
}bssid_record_t;


static void recv_byte(unsigned char pos, unsigned char byte);


static unsigned char g_bssid_record_sn;
static bssid_record_t g_bssid_record[BSSID_RECODR_NUM];

static unsigned char g_res_len;
static unsigned char g_res_data_map[128];
static unsigned char g_res_data[128];

static char g_lock;
static char g_complete;
static unsigned char g_cur_bssid[6];

//broadcast
static unsigned char g_bc_byte_index[2];
static unsigned char g_bc_byte_data[2][3];

_tag static unsigned short _mtstrlen(const char *s)
{
    const char *sc;

    for(sc = s; *sc != '\0'; ++sc)
        /* nothing */;
    return (unsigned short)(sc - s);
}

_tag static void *_mtmemset(void *s, int c, unsigned short count)
{
    char *xs = s;
    while(count--)
        *xs++ = (char)c;
    return s;
}

_tag static void *_mtmemcpy(void *dest, const void *src, unsigned short count)
{
    char *tmp = dest;
    const char *s = src;

    while(count--)
        *tmp++ = *s++;
    return dest;
}

_tag static unsigned short _mtmemcmp(const void *cs, const void *ct, unsigned short count)
{
    const unsigned char *su1, *su2;
    unsigned short res = 0;

    for(su1 = cs, su2 = ct; 0 < count; ++su1, ++su2, count--)
        if((res = *su1 - *su2) != 0)
            break;
    return res;
}

_tag void mtlink_init()
{

}

_tag void mtlink_clear()
{
    g_lock = 0;
    g_complete = 0;
    g_res_len = 0;
    _mtmemset(g_res_data_map, 0, sizeof(g_res_data_map));
}

_tag static bssid_record_t *find_bssid_record(const unsigned char *bssid)
{
    int i;
    bssid_record_t *record;
    for(i = 0; i < BSSID_RECODR_NUM; i++)
    {
        record = (bssid_record_t *)&g_bssid_record[i];

        if(_mtmemcmp(bssid, record->bssid, 6) == 0)
        {
            return record;
        }
    }

    return NULL;
}

_tag static bssid_record_t *find_old_bssid_record()
{
    int i;
    unsigned char max_sn_diff = 0;
    int maxi = 0;
    unsigned char sn_diff = 0;
    bssid_record_t *record;
    for(i = 0; i < BSSID_RECODR_NUM; i++)
    {
        record = (bssid_record_t *)&g_bssid_record[i];
        sn_diff = g_bssid_record_sn - record->sn;
        if(sn_diff > max_sn_diff)
        {
            maxi = i;
            max_sn_diff = sn_diff;
        }
    }

    return &g_bssid_record[maxi];
}

_tag static void lock(const unsigned char *bssid)
{
    g_lock = 1;
    _mtmemcpy(g_cur_bssid, bssid, 6);
}

_tag static void recv_len(unsigned char len)
{
    g_res_len = len;
    mtlog("len:%d", len);
}

_tag static char broadcast_recv(unsigned char value, int fromsta)
{
    if(value == 0)
    {
        return 0;
    }

    //pos
    if(!(value & (1 << 7)))
    {
        g_bc_byte_index[fromsta] = 0;
        g_bc_byte_data[fromsta][g_bc_byte_index[fromsta]++] = value;
        return 0;
    }

    if(g_bc_byte_data[fromsta][0] == 0)
    {
        return 0;
    }

    g_bc_byte_data[fromsta][g_bc_byte_index[fromsta]++] = value;

    char recved_data = 0;
    //
    if(g_bc_byte_index[fromsta] >= 3)
    {
        //Уͨ
        if(((g_bc_byte_data[fromsta][0] ^ g_bc_byte_data[fromsta][1]) & 0x7f) == (g_bc_byte_data[fromsta][2] & 0x7f))
        {
            recved_data = 1;

            char gt128 = g_bc_byte_data[fromsta][0] & (1 << 6);
            unsigned char pos = (g_bc_byte_data[fromsta][0] & 0x3f) - 1;
            if(gt128)
            {
                value = g_bc_byte_data[fromsta][1];
            }
            else
            {
                value = (g_bc_byte_data[fromsta][1] & 0x7f);
            }

            if(pos == 0)
            {
                recv_len(value);
            }
            else
            {
                //mtlog("b:%d", fromsta);
                recv_byte(pos - 1, value);
            }
        }
        g_bc_byte_data[fromsta][0] = 0;
    }

    return recved_data;
}

_tag static void recv_byte(unsigned char pos, unsigned char byte)
{
    if(g_complete)
    {
        return;
    }

    if(pos < sizeof(g_res_data))
    {
        if(!g_res_data_map[pos])
        {
            g_res_data[pos] = byte;
            g_res_data_map[pos] = 1;
        }
    }

    if(g_res_len == 0)
    {
        return;
    }

    char check = 0;
    int i;
    for(i = 0; i < g_res_len; i++)
    {
        if(!g_res_data_map[i])
        {
            return;
        }
        check ^= g_res_data[i];
    }

    //Уʧ
    if(check != 0)
    {
        mtlog("chk fail");
        g_res_len = 0;
        _mtmemset(g_res_data_map, 0, sizeof(g_res_data_map));
        return;
    }

    //
    g_complete = 1;
    g_res_data[g_res_len - 1] = '\0';
}

_tag static char muticast_recv(const unsigned char *data, int fromsta)
{
    if(data[0] == 0x7e && data[2] == 0xaa)
    {
        recv_len(data[1]);
        return 1;
    }

    if(!g_lock)
    {
        return 0;
    }

    //mtlog("m%d", fromsta);

    recv_byte(data[0], data[1]);
    recv_byte(data[0] + 1, data[2]);

    return 1;
}

_tag mtlink_status_t mtlink_recv(const void *frame, unsigned short len)
{
    char isbroadcast;
    const IEEE80211DataFrame_t *df = (const IEEE80211DataFrame_t *)frame;

    if(df->framectrl.type != 2)
    {
        return MTLINK_STATUS_CONTINUE;
    }

    if(df->framectrl.subtype != 0 && df->framectrl.subtype != 8)
    {
        return MTLINK_STATUS_CONTINUE;
    }

    const unsigned char *bssid;
    const unsigned char *addr;
    const unsigned char bmac[6] = {0xff, 0xff, 0xff, 0xff, 0xff, 0xff};
    int fromsta = 0;

    //Station
    if(df->framectrl.toDS == 1 && df->framectrl.FromDS == 0)
    {
        bssid = df->raddr;
        addr = df->saddr;
        fromsta = 1;
    }
    //Client
    else if(df->framectrl.toDS == 0 && df->framectrl.FromDS == 1)
    {
        fromsta = 0;
        bssid = df->taddr;
        addr = df->raddr;
    }
    else
    {
        return MTLINK_STATUS_CONTINUE;
    }

    //鲥ַ
    if(addr[0] == 0x01 && addr[1] == 0x00 && addr[2] == 0x5e)
    {
        isbroadcast = 0;
    }
    else if(_mtmemcmp(bmac, addr, 6) == 0)
    {
        isbroadcast = 1;
    }
    else
    {
        return MTLINK_STATUS_CONTINUE;
    }

    //mac
    if(g_lock && _mtmemcmp(bssid, g_cur_bssid, 6) != 0)
    {
        return MTLINK_STATUS_CONTINUE;
    }

    bssid_record_t *record = find_bssid_record(bssid);
    if(record == NULL)
    {
        mtlog("+");
        record = find_old_bssid_record();
        record->len_offset[0] = 0xffff;
        record->len_offset[1] = 0xffff;
        _mtmemcpy(record->bssid, bssid, 6);
    }
    record->sn = g_bssid_record_sn++;

    //㳤offset
    if(len < record->len_offset[fromsta])
    {
        record->len_offset[fromsta] = len;
    }

    unsigned short datalen = len - record->len_offset[fromsta];

    char recv_valid_data = 0;
    g_complete = 0;

    //㲥
    if(isbroadcast)
    {
        if(datalen > 256)
        {
            return MTLINK_STATUS_CONTINUE;
        }

        recv_valid_data = broadcast_recv((unsigned char)datalen, fromsta);
    }
    //ಥ
    else
    {

        if(datalen > 1)
        {
            return MTLINK_STATUS_CONTINUE;
        }
        recv_valid_data = muticast_recv(addr + 3, fromsta);
    }

    if(recv_valid_data)
    {
        if(!g_lock)
        {
            mtlog("lock");
            lock(bssid);
            return MTLINK_STATUS_LOCK;
        }
        else
        {
            if(g_complete)
            {
                mtlog("end");
                return MTLINK_STATUS_COMPLETE;
            }
            return MTLINK_STATUS_RECV_VALID_DATA;
        }
    }

    return MTLINK_STATUS_CONTINUE;
}

_tag void mtlink_get_res(mtlink_res_t *res)
{
    if(g_complete)
    {
        unsigned char len = 0;
        _mtmemcpy(res->app_ip, g_res_data + len, 4);
        len += 4;
        res->app_port = ((unsigned short)g_res_data[len]) << 8 | g_res_data[len + 1];
        len += 2;
        res->app_port = res->app_port + 1; //APPڷ͵Ķ˿ں+1ϽUDP
        res->pwd = (char *)g_res_data + len;
        len = (unsigned char)(len + _mtstrlen(res->pwd) + 1);
        res->ssid = (char *)res->pwd + _mtstrlen(res->pwd) + 1;
        len = (unsigned char)(len + _mtstrlen(res->ssid) + 1);

        if(len < g_res_len)
        {
            res->option = g_res_data[len++];
        }
        else
        {
            //res->option = 0x00; //ǳ
            res->option = 0x01; //
        }
        mtlog("option:%02x", res->option);
    }
}

