//
//  MachtalkSDK.h
//  MachtalkSDK
//
//  Created by Aibo on 15/10/19.
//  Copyright © 2015年 Aibo. All rights reserved.
//

#import <Foundation/Foundation.h>
#import "MachtalkSDKConstant.h"
#import "MachtalkSDKDelegate.h"

@class Device;
@class User;
//@class Catalog;
//@class ProductCategory;
@class TimerTask;
@class DeviceAidStatisticDataParam;
@class AddDeviceUtil;

@interface MachtalkSDK : NSObject

@property (nonatomic, assign) BOOL isSDKStarted;

+(instancetype)Instance;


#define SDK_VERSION @"Version:2.1.0.0"

#pragma mark - SDK初始化相关接口
/**
 *  启动SDK
 *
 *  @param platformID  云平台ID，需要厂商向云平台提供商提出申请
 *  @param appID 产品ID，需要厂商向云平台提供商提出申请
 *  @param productCatalogId 产品列表对应的Id,用于从服务器下载相应的产品列表。如果设置成nil,则不下载
 *
 *  @return YES:启动成功 NO:启动失败
 */
-(BOOL)startSDK:(NSString*)platformID appID:(NSString*)appID productCatalogId:(NSString*)productCatalogId;

/**
 *  停止SDK
 *
 *  @return YES:停止 成功 NO:停止失败
 */
-(BOOL)stopSDK;

/**
 *  开始打印log
 */
-(void)startLog;

/**
 *  停止打印log
 */
-(void)stopLog;

/**
 *  添加SDK代理
 *
 *  @param delegate
 */
- (void)setSdkDelegate:(id<MachtalkSDKDelegate>)delegate;

/**
 *   移除SDK代理
 *
 *  @param delegate 
 */
- (void)removeSdkDelegate:(id<MachtalkSDKDelegate>)delegate;

/**
 *   设置长连接服务器和端口
 *  不设置的话。默认连接智城云官方服务器
 *
 *  @param url 服务器地址 ip或者域名
 *  @param port 端口 如果不需修改 传入nil
 */
- (void)setServerNls:(NSString*)url port:(NSString*)port;

/**
 *   设置HTTP服务器
 *  不设置的话。默认连接智城云官方服务器
 *
 *  @param url 服务器地址 ip或域名
 */
- (void)setServerHttp:(NSString*)url;



#pragma mark - 用户系统相关接口


#pragma mark - 用户协议

/**
 *   退出登录
 */
- (void)userLogout;

/**
 *  连接登录服务器
 *
 *  @param user       用户名、手机号或者邮箱地址
 *  @param pwd        登录密码
 */
- (void)userLogin:(NSString*)user pwd:(NSString*)pwd;


#pragma mark - 用户接口

/**
 *  用户注册
 *
 *  @param user 用户信息
 *  @param token 服务器返回的校验码
 */
- (void)userRegister:(User*)user token:(NSString*)token;

/**
 *  查询用户名是否已经被注册过
 *
 *  @param type    用户注册方式：手机还是邮箱
 *  @param value   EMAIL对应邮箱 TELEPHONE对应电话号码
 */
- (void)checkUserIsRegistered:(UserType)type value:(NSString*)value;

/**
 *  请求服务器向用户发送验证码
 *
 *  @param type        用户注册方式：手机还是邮箱
 *  @param regPhone    email对应邮箱值 telephone对应手机号
 *  @param valcodeType 验证码类型
 */
-(void)sendValcode:(UserType)type value:(NSString*)value valcodeType:(ValCodeType)valcodeType;

/**
 *  请求服务器向用户发送校验码
 *  @param type        用户注册方式：手机还是邮箱
 *  @param value       email对应邮箱值 telephone对应手机号
 *  @param valcode     收到的验证码
 *  @param valcodeType 验证码类型
 */
-(void)checkValCode:(UserType)type value:(NSString*)value valcode:(NSString*)valcode valcodeType:(ValCodeType)valcodeType;

/**
 *  获取用户信息
 */
- (void)getUserInfo;

/**
 *  修改用户信息
 *
 *  @param user 要设置的用户信息
 */
- (void)editUserInfo:(User*)user;

/**
 *  修改用户密码
 *
 *  @param oldPwd 旧密码
 *  @param newPwd 新密码
 */
- (void)modifyUserPassword:(NSString*)oldPwd newPwd:(NSString*)newPwd;

/**
 *  找回用户密码
 * 
 *  @param type      用户注册方式：手机还是邮箱
 *  @param value     用户手机号码或者邮箱
 *  @param pwd       密码
 *  @param token     校验码
 */
- (void)findUserPassword:(UserType)type value:(NSString*)value pwd:(NSString*)pwd token:(NSString*)token;

/**
 * 用户修改手机号码
 *
 * @param telephone 手机号
 * @param valCode 验证码
 * @return
 */
- (void)modPhone:(NSString*)telephone valCode:(NSString*)valCode;

/**
 * 用户修改邮箱
 *
 * @param email 新邮箱
 * @param valCode 验证码
 * @return
 */
- (void)modEmail:(NSString*)email valCode:(NSString*)valCode;

/**
 *  登陆后修改用户头像
 *
 *  @param userPhone 图片名称
 *  @param base64    图片的二进制字节
 */
-(void)editUserImage:(NSString*)filename base64:(NSString*)base64;

/**
 *  从服务器下载头像文件到本地
 *
 *  @param fileUrl 图片存在的网络url路径，从editUserImage的回调函数中可以得到，或者从user.avatar中得到
 */
-(void)getUserImage:(NSString*)fileUrl;

#pragma mark - App相关接口

/**
 *  获取app启动页（广告页）
 *
 *  @param appId
 */
- (void)getAppStartPage:(NSString*)appId;

/**
 * 请求APP基本信息
 *
 * @param appId APP ID
 * @param catalogId 当前APP的残品分类层级编号
 * @return
 */
+(void)getAppBaseInfo:(NSString*)appId catalogId:(NSString*)catalogId;

/**
 * 查询APP最新版本信息
 *
 * @param appId APP ID
 * @return
 */
+(void)getAppLatestVersion:(NSString*)appId;

/**
 *  App提交对App的意见反馈
 *
 *  @param appId   appid
 *  @param feedbackContent 反馈内容
 *  @param userContact 用户的联系方式（可选，不提供可设置成nil）
 */
- (void)commitFeedbackAboutApp:(NSString*)appId feedbackContent:(NSString*)content userContact:(NSString*)contact;

/**
 *  订阅或者取消订阅官方消息
 *
 *  @param subscribe YES: 订阅，NO:取消订阅
 */
- (void)officialMessageSubscribe:(BOOL)subscribe;

/**
 *  查询官方消息详情
 *
 *  @param messageId 官方消息id
 */
- (void)getOfficialMessageDetail:(NSString*)messageId;

/**
 *  获取错误码对应的错误信息
 *
 *  @param errorCode 错误码
 */
-(void)getErrorCodeInfo:(NSString*)errorCode;



#pragma mark - 设备相关接口

#pragma mark - 设备协议

/**
 *  获取设备当前状态
 *
 *  @param device 当前设备
 */
- (void)queryDeviceStatus:(NSString*)deviceId;

/**
 *  操作设备
 *
 *  @param device 设备
 *  @param dvidArray   要修改的功能属性id的数组
 *  @param valueArray  功能属性要修改成的值的数组
 */
- (void)sendOrder:(Device*)device dvid:(NSMutableArray*)dvidArray value:(NSMutableArray*)valueArray;

/**
 *  配置设备WiFi
 *
 *  @param addDeviceUtil     addDeviceUtil的对象
 *  @param scanCode 设备的类型吗，对于需要扫码添加的设备需要提供，对于无需扫码的设备请设置成：NO_SCAN_CODE
 *  @param type     目前仅支持SetDeviceWiFiType和SET_WIFI_SMART
 *  @param addWiFiType  添加wifi的来源方式 是添加WIFI（SET_WIFI_ADD）还是更新WIFI（SET_WIFI_UPDATE）
 */
- (void)setDeviceWiFi:(AddDeviceUtil*)addDeviceUtil scanCode:(NSString*)scanCode wifiType:(SetDeviceWiFiType)type addWiFiType:(SetAddWiFiType)addWiFiType;

/**
 *  停止配置设备WiFi
 *
 *  @param addDevicUtil          添加的对象
 *  @param type                  设置wifi的类型
 */
-(void)stopConfigWiFi:(AddDeviceUtil*)addDevicUtil wifiType:(SetDeviceWiFiType)type;

/**
 *  搜索同一局域网内的设备
 *
 *  @param period 搜索间隔，即多长时间搜索一次，如果设置的数值小于等于0，则只搜索一次
 */
- (void)startSearchLanDevices:(long)period;

/**
 *  停止搜索同一局域网内的设备，没有回调函数
 */
- (void)stopSearchLanDevices;

/**
 *  与同一局域网下的设备建立连接
 *
 *  @param deviceId 设备id
 *  @param deviceIp 同一局域网下分配给设备的ip地址，可以通过搜索同一局域网下设备返回的信息中获得
 *  @param deviceIp 是否为1.0设备 默认为NO
 */
- (void)connectLanDevice:(NSString*)deviceId deviceIp:(NSString*)deviceIp isOld:(BOOL)isOld;


#pragma mark - 设备接口

/**
 *  请求设备固件的更新
 *
 *  @param message 接收到的服务器推送的升级消息，可从回调函数 onReceiveModuleUpdateMessage中获得
 */
- (void)requestDeviceModuleUpgrade:(UpdateMessage*)message;

/**
 *  获取设备添加层级列表
 *
 *  @param productCatalogId 智诚云分配给厂商的产品列表id
 */
- (void)downloadProductCatalogInfo:(NSString*)productCatalogId;

/**
 *  获取设备引导信息
 *
 *  @param deviceModel 设备对应的型号，如果有设备列表，可以从获取的产品列表中对应的设备信息中获取
 *  @param mode        添加方式 1:smart模式,2:ap模式
 */
- (void)getDeviceGuideInfo:(NSString*)deviceModel addMode:(NSString*)mode;

/**
 *  获取设备引导图片
 *
 *  @param pictureUrl 设备引导的某一步所对应的图片下载地址，这个可以在获取到的设备引导信息中取得
 */
- (void)getDeviceGuidePicture:(NSString*)pictureUrl;

/**
 *  获取设备分享码
 *
 *  @param deviceId 设备ID
 */
- (void)getDeviceShareCode:(NSString*)deviceId;

/**
 *  绑定设备
 *
 *  @param did   设备id
 *  @param pin   设备pin
 *  @param model 设备型号,没有设备类型时设置为nil
 * @param key 如果是发现设备就是设备返回的key 否者传入nil
 * @param timeStr 如果是发现设备就是设备发现时的timeStr 否者传入nil
 */

- (void)bindDevice:(NSString*)deviceId pin:(NSString*)pin model:(NSString*)model key:(NSString*)key timeStr:(NSString*)timeStr;

/**
 *  分享二维码绑定设备
 *
 *  @param shareCode  分享二维码
 */
- (void)bindDeviceByShareCode:(NSString*)shareCode;

/**
 *  解绑设备
 *
 * @param device 设备对象
 * @param type 解绑类型 主人解绑、主人解绑客人 (可为nil)
 * @param uid 根据type不同，uid不同 主人解绑(新主人ID)，主人解绑客人(客人ID) (可为nil)
 */
- (void)unbindDevice:(Device*)device unbindType:(UnbindType)type uid:(NSString*)uid;

/**
 *  修改设备名称
 *
 *  @param did  需要修改的设备id
 *  @param newName 修改后的设备显示名称
 */
- (void)modifyDeviceName:(NSString*)did newName:(NSString*)newName;

/**
 *  查询设备列表
 */
- (void)queryDeviceList;

/**
 *  重置设备顺序
 *
 *  @param deviceOrderDic 设备id以及对应的位置的键值
 */
- (void)reOrderDevice:(NSDictionary*)deviceOrderDic;

/**
 *  获取用户离线期间的消息
 */
- (void)queryOffLineMessages;

/**
 *  获取设备用户信息
 *
 *  @param deviceId 设备Id
 */
- (void)queryDeviceUsersInfo:(NSString*)deviceId;

/**
 *  获取设备属性详情
 *
 *  @param deviceModel 设备model
 */
- (void)queryDeviceDescription:(NSString*)deviceModel;


/**
 *  转让设备的主控人权限
 *
 *  @param deviceId 设备id
 *  @param userName 指定的成为设备主控人的用户（用户可以通过相应接口获取设备的用户信息，从中得到指定的用户信息）
 */
- (void)deviceHostTransfer:(NSString*)deviceId userName:(NSString*)userName;

/**
 *  建立设备分组
 *
 *  @param groupName     要建立的分组名称
 *  @param deviceIdList  要建立的分组所包含的设备id列表
 */
- (void)addDeviceGroup:(NSString*)groupName deviceIdList:(NSMutableArray*)deviceIdList;

/**
 *  修改设备分组名称
 *
 *  @param groupId   要修改的分组id（建立分组时由云服务器分配并返回给用户）
 *  @param groupName 修改后的分组名称
 */
- (void)modifyDeviceGroupName:(NSString*)groupId groupName:(NSString*)groupName;

/**
 *  保存设备分组
 *
 *  @param groupId      要修改的分组id（建立分组时由云服务器分配并返回给用户）
 *  @param deviceIdList 整理后的设备id列表
 */
- (void)saveDeviceGroup:(NSString*)groupId deviceIdList:(NSMutableArray*)deviceIdList;

/**
 *  整理后的设备id列表
 *
 *  @param groupId 要删除的分组id（建立分组时由云服务器分配并返回给用户）
 */
- (void)deleteDeviceGroup:(NSString*)groupId;

/**
 *  查询用户所有设备固件版本信息
 */
- (void)queryAllDevicesModuleVersionInfo;

/**
 * 获取app最新版本信息
 */
-(void)getAppLatestVersion:(NSString*)appId;

/**
 *  查询某个设备的固件版本信息
 *
 *  @param deviceId 设备id
 */
- (void)queryDeviceModuleVersionInfo:(NSString*)deviceId;

/**
 *  提交对于设备的意见反馈
 *
 *  @param deviceModel 设备类型
 *  @param content     智诚云分配给app的id
 *  @param contact     反馈内容
 *  @param appId       用户的联系方式（可选，不提供可设置成nil）
 */
- (void)commitFeedbackAboutDevice:(NSString*)deviceModel feedbackContent:(NSString*)content userContact:(NSString*)contact appId:(NSString*)appId;

/**
 * 获取设备某个属性的统计数据
 *
 */
-(void)getDeviceAidStatisticData:(DeviceAidStatisticDataParam*)param;

/**
 * 查询定时任务列表
 * @param deviceId 设备id
 */
-(void)queryTimerTaskList:(NSString *)deviceId;

/**
 * 查询定时任务详情
 * @param taskId 任务id
 */
-(void)queryTimerTaskInfo:(NSString *)taskId;

/**
 * 创建定时任务
 * @param taskInfo 任务详细信息
 * @param command 任务命令类型
 */
-(void)createTimerTask:(TimerTask *)taskInfo cmd:(NSString *)command;

/**
 * 修改定时任务
 * @param taskInfo 任务详细信息, taskId必须设置
 * @param command 任务命令类型
 */
-(void)modifyTimerTask:(TimerTask *)taskInfo cmd:(NSString *)command;

/**
 * 删除定时任务
 * @param taskId 任务id
 */
-(void)removeTimerTask:(NSString *)taskId;

/**
 * 删除设备所有定时任务
 * @param did 设备id
 */
-(void)removeAllTimerTask:(NSString *)did;

/**
 * 获得设备定时任务数量
 * @param did 设备id
 */
-(void)getTimerTaskCount:(NSString *)did;

/**
 * 设备通用数据存储
 * @param devId 设备ID
 *        data  要保存的设备属性值集合
 */
-(void)saveDeviceDatas:(NSString *)devId data:(NSDictionary *)data;

/**
 * 设备通用数据统计
 * @param devId 设备ID
 */
-(void)getDeviceDatas:(NSString *)devId;


@end

